/*
 * Decompiled with CFR 0.152.
 */
package de.alamos.ioespa.services.ioprint;

import de.alamos.firemergency.ioprint.responses.EPrinterError;
import de.alamos.firemergency.ioprint.responses.EPrinterResponse;
import de.alamos.firemergency.ioprint.responses.EPrinterStatus;
import de.alamos.firemergency.ioprint.responses.PrinterStatusResponse;
import de.alamos.ioespa.data.printing.JobInfo;
import de.alamos.ioespa.helper.ExecuteCommandHelper;
import de.alamos.ioespa.helper.LoggerHelper;
import de.alamos.ioespa.helper.exceptions.GenericException;
import de.alamos.ioespa.services.config.ConfigurationService;
import de.alamos.ioespa.services.ioprint.PrinterRulesService;
import de.alamos.ioespa.services.ioprint.ResumingService;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@Component
@Profile(value={"print"})
@SuppressFBWarnings(value={"DM_CONVERT_CASE"})
public class PrinterService {
    public static final int PRINTER_INFO_LINE_LENGTH = 4;
    public static final int LINE_INDX_3 = 3;
    public static final int LINE_INDX_2 = 2;
    public static final int MAX_TIME_JOB_SHOULD_BE_IN_QUEUE = 45;
    public static final int WAIT_JOB_SENDING_IN_SEC = 5;
    private final Logger logger = LoggerHelper.getLogger(this.getClass());
    private final ExecuteCommandHelper commandHelper;
    private final PrinterRulesService printerRulesService;
    private List<String> disabledPrinters;
    @Value(value="${supportMultiplePrinters}")
    private boolean supportMultiplePrinters;
    @Value(value="${printer.model}")
    private String name;
    @Value(value="${printer.driverPath}")
    private String driverPath;
    private final ConfigurationService configurationService;
    private final ResumingService resumingService;
    private static final List<String> PRINTING = List.of("job-incoming", "job-data-insufficient", "job-outgoing", "job-interpreting", "job-transforming", "job-printing");
    private static final List<String> QUEUED = List.of("job-queued", "job-queued-for-marker", "queued-in-device", "none");
    private static final List<String> OFFLINE = List.of("printer-stopped-partly", "printer-stopped", "service-off-line");
    private static final List<String> CANCELED = List.of("job-canceled-by-user", "job-canceled-by-operator", "job-canceled-at-device", "aborted-by-system");
    private static final List<String> FAILED = List.of("unsupported-compression", "compression-error", "unsupported-document-format", "document-format-error", "document-access-error", "submission-interrupted", "job-completed-with-warnings", "job-completed-with-errors");
    private static final List<String> COMPLETED = List.of("job-completed-successfully", "processing-to-stop-point");
    private static final List<String> HAS_PROBLEM = List.of("resources-are-not-ready");

    public PrinterService(ExecuteCommandHelper commandHelper, PrinterRulesService printerRulesService, ConfigurationService configurationService, ResumingService resumingService) {
        this.commandHelper = commandHelper;
        this.printerRulesService = printerRulesService;
        this.configurationService = configurationService;
        this.resumingService = resumingService;
    }

    public Optional<String> createPrinter() {
        Optional usbUrl = this.commandHelper.getPrinterUrl();
        if (usbUrl.isEmpty()) {
            return Optional.empty();
        }
        this.commandHelper.createPrinter(this.name, (String)usbUrl.get(), this.driverPath);
        this.printerRulesService.createPrinterRules();
        this.restartCups();
        return Optional.of(this.name);
    }

    public void removePrinter(String printerName) {
        this.commandHelper.removePrinter(printerName);
    }

    public void removeAllPrinters() {
        try {
            this.commandHelper.getPrintersList().forEach(arg_0 -> this.removePrinter(arg_0));
        }
        catch (Exception e) {
            this.logger.error("Can not get Printers List", (Throwable)e);
        }
    }

    public List<String> getDisabledPrinters() {
        if (this.supportMultiplePrinters) {
            if (this.disabledPrinters == null) {
                this.loadFromFile();
            }
            return this.disabledPrinters;
        }
        return new ArrayList<String>();
    }

    public void disablePrinter(String printerName, boolean isConnected) {
        if (!this.supportMultiplePrinters) {
            return;
        }
        if (this.disabledPrinters == null) {
            this.disabledPrinters = new ArrayList();
        }
        if (isConnected) {
            this.disabledPrinters.add(printerName.trim());
        } else {
            this.disabledPrinters.remove(printerName.trim());
        }
        this.saveToFile();
    }

    public boolean isEnable(String name) {
        return !this.getDisabledPrinters().contains(name);
    }

    @SuppressFBWarnings(value={"DM_DEFAULT_ENCODING"})
    private void saveToFile() {
        try {
            Path printers = this.getDisabledPrintersFilePath(true);
            if (printers == null) {
                this.logger.warn("Cannot save DisabledPrinters, file not found.");
                return;
            }
            try (FileWriter writer = new FileWriter(printers.toString(), false);){
                for (String s : this.disabledPrinters) {
                    writer.write(s);
                    writer.write("\n");
                }
            }
        }
        catch (IOException e) {
            this.logger.error("Cannot save DisabledPrinters", (Throwable)e);
        }
    }

    private void loadFromFile() {
        this.logger.trace("loadFromFile()");
        this.disabledPrinters = new ArrayList();
        try {
            Path printers = this.getDisabledPrintersFilePath(false);
            if (printers == null) {
                this.logger.warn("Cannot load DisabledPrinters, file not found.");
                return;
            }
            try (BufferedReader reader = Files.newBufferedReader(printers);){
                String line;
                while ((line = reader.readLine()) != null) {
                    this.disabledPrinters.add(line.trim());
                }
            }
        }
        catch (IOException e) {
            this.logger.error("Cannot load DisabledPrinters", (Throwable)e);
        }
    }

    private Path getDisabledPrintersFilePath(boolean createIfNotExist) throws IOException {
        File file = new File("config/disabledPrinters");
        try {
            if (Files.exists(file.toPath(), new LinkOption[0])) {
                return Paths.get(file.toURI());
            }
        }
        catch (Exception e) {
            this.logger.error("Error geting File {}", (Object)file.getAbsolutePath(), (Object)e);
        }
        if (createIfNotExist) {
            return this.createDisabledPrintersFile();
        }
        return null;
    }

    @Nullable
    private Path createDisabledPrintersFile() {
        File file = new File("config/disabledPrinters");
        Path printers = file.toPath();
        if (!Files.exists(printers, new LinkOption[0])) {
            try {
                Files.createFile(printers, new FileAttribute[0]);
                this.logger.info("File {} created", (Object)printers);
                return printers;
            }
            catch (Exception ex) {
                this.logger.error("File {} could not be created", (Object)printers, (Object)ex);
                return null;
            }
        }
        return printers;
    }

    protected File createTmp(byte[] pdf) {
        try {
            File tmp = File.createTempFile("printing" + Arrays.hashCode(pdf), ".pdf");
            this.logger.debug("TempFile created at path {}", (Object)tmp.getAbsolutePath());
            Files.write(tmp.toPath(), pdf, new OpenOption[0]);
            return tmp;
        }
        catch (IOException e) {
            this.logger.error("Error creating tmp file", (Throwable)e);
            return null;
        }
    }

    public boolean isRestartingServicesDone() {
        return ExecuteCommandHelper.isRestartingServicesDone();
    }

    public List<PrinterStatusResponse> getPrinterStatusList() {
        return this.getPrintersList().stream().map(arg_0 -> this.readPrinterStatus(arg_0)).collect(Collectors.toList());
    }

    public Map<String, JobInfo> print(byte[] pdf, int copies) throws GenericException {
        this.logger.trace("print byte[]'{}' hash, {} copies", (Object)Arrays.hashCode(pdf), (Object)copies);
        return this.print(this.createTmp(pdf), copies);
    }

    public Map<String, JobInfo> print(File pdf, int copies) throws GenericException {
        this.logger.trace("print File '{}', {} copies", (Object)(pdf != null ? pdf.getAbsolutePath() : "null"), (Object)copies);
        List enabledPrinters = this.getEnabledPrinters();
        if (enabledPrinters.isEmpty()) {
            this.logger.error("List enabled printers is Empty");
            throw new GenericException("No printers");
        }
        return this.print(pdf, copies, enabledPrinters);
    }

    @Nullable
    private Map<String, JobInfo> print(File pdf, int copies, List<String> enabledPrinters) {
        this.logger.trace("print File {} in {} enabledPrinters, {} copies", new Object[]{pdf != null ? pdf.getAbsolutePath() : "null", enabledPrinters.size(), copies});
        if (pdf == null) {
            this.logger.error("Tmp file is empty or not created. Nothing to print");
            return null;
        }
        try {
            return ((Stream)enabledPrinters.stream().parallel()).collect(Collectors.toMap(c -> c, c -> this.print(pdf, copies, c)));
        }
        catch (Exception e) {
            this.logger.error("Error during printing!", (Throwable)e);
            return null;
        }
    }

    public JobInfo print(File file, int copies, String printerName) {
        this.logger.trace("print File {} in printer {}, {} copies", new Object[]{file != null ? file.getAbsolutePath() : "null", printerName, copies});
        try {
            if (file == null) {
                return new JobInfo(EPrinterResponse.Failed, Optional.empty());
            }
            Optional jobName = this.commandHelper.print(copies, printerName, file, this.configurationService.loadAppConfig().isDuplexOn());
            if (jobName.isPresent()) {
                try {
                    TimeUnit.SECONDS.sleep(5L);
                }
                catch (InterruptedException e) {
                    throw new GenericException((Exception)e);
                }
                EPrinterResponse ePrinterResponse = this.askStatusJob(printerName, (String)jobName.get());
                this.logger.info("For {} printer EPrinterResponse is {}", (Object)printerName, (Object)ePrinterResponse.name());
                return new JobInfo(ePrinterResponse, jobName);
            }
            this.logger.warn("For {} printer EPrinterResponse is Failed", (Object)printerName);
            return new JobInfo(EPrinterResponse.Failed, Optional.empty());
        }
        catch (GenericException | IOException e) {
            this.logger.warn("For {} printer EPrinterResponse is Unknown", (Object)printerName, (Object)e);
            return new JobInfo(EPrinterResponse.Unknown, Optional.empty());
        }
    }

    private List<String> getPrintersList() {
        try {
            return this.commandHelper.getPrintersList();
        }
        catch (Exception e) {
            this.logger.error("Error getting list of printers ", (Throwable)e);
            return new ArrayList<String>();
        }
    }

    private List<String> getEnabledPrinters() {
        return this.getPrintersList().stream().filter(arg_0 -> this.isEnable(arg_0)).collect(Collectors.toList());
    }

    private PrinterStatusResponse readPrinterStatus(String name) {
        try {
            String answer = this.commandHelper.getPrinterStatus(name);
            return this.parsePrinterStatus(name, answer);
        }
        catch (IOException e) {
            return new PrinterStatusResponse(name, EPrinterStatus.Unknown, EPrinterError.Unknown, Boolean.valueOf(this.isEnable(name)));
        }
    }

    @NotNull
    public PrinterStatusResponse parsePrinterStatus(String name, String answer) {
        if (!StringUtils.hasLength((String)answer)) {
            return new PrinterStatusResponse(name, EPrinterStatus.Unknown, EPrinterError.Unknown, Boolean.valueOf(this.isEnable(name)));
        }
        String[] printerInfo = answer.split(" ");
        if (printerInfo.length <= 4) {
            return new PrinterStatusResponse(name, EPrinterStatus.Unknown, EPrinterError.Unknown, Boolean.valueOf(this.isEnable(name)));
        }
        String answerStatus = (printerInfo[2] + " " + printerInfo[3]).toLowerCase();
        EPrinterStatus ePrinterStatus = EPrinterStatus.Unknown;
        EPrinterError ePrinterError = EPrinterError.Unknown;
        if (answerStatus.contains("no suitable destination host found")) {
            ePrinterStatus = EPrinterStatus.Has_Error;
            ePrinterError = EPrinterError.Printer_error;
            return new PrinterStatusResponse(name, ePrinterStatus, ePrinterError, Boolean.valueOf(this.isEnable(name)));
        }
        if (answerStatus.contains("idle")) {
            ePrinterStatus = EPrinterStatus.Ready;
        }
        if (answerStatus.contains("faulted")) {
            ePrinterStatus = EPrinterStatus.Has_Error;
        }
        if (answerStatus.contains("paper misfeed")) {
            ePrinterError = EPrinterError.No_Paper;
        }
        if (answerStatus.contains("rendering completed") || answerStatus.contains("low ink")) {
            ePrinterError = EPrinterError.Low_Toner;
        }
        if (answerStatus.contains("now printing")) {
            ePrinterError = this.parsePrintingError(answer, name);
            EPrinterStatus ePrinterStatus2 = ePrinterStatus = EPrinterError.Other.equals((Object)ePrinterError) ? EPrinterStatus.Has_Error : EPrinterStatus.Printing;
        }
        if (answerStatus.contains("offline") || answerStatus.contains("disabled") || answerStatus.contains("waiting for printer to become available") || answerStatus.contains("unable to locate printer")) {
            ePrinterError = EPrinterError.Offline;
            this.resumingService.enableResuming();
        }
        return new PrinterStatusResponse(name, ePrinterStatus, ePrinterError, Boolean.valueOf(this.isEnable(name)));
    }

    private EPrinterError parsePrintingError(String answer, String printer) {
        this.logger.trace("parsePrintingError answer-'{}',printer-'{}'", (Object)answer, (Object)printer);
        int jobNameStart = answer.indexOf("now printing") + "now printing ".length();
        int jobNameEnd = answer.indexOf(" ", jobNameStart);
        if (jobNameEnd == -1) {
            return EPrinterError.Unknown;
        }
        String jobName = answer.substring(jobNameStart, jobNameEnd).trim();
        try {
            Optional jobStartingTime = this.commandHelper.getJobStartingTime(printer, jobName);
            if (jobStartingTime.isPresent() && ChronoUnit.SECONDS.between((Temporal)jobStartingTime.get(), ZonedDateTime.now()) > 45L) {
                this.logger.info("Job {} printing > 20 sec. Unknown error", (Object)jobName);
                return EPrinterError.Other;
            }
        }
        catch (IOException e) {
            this.logger.warn("Error parsing job starting time", (Throwable)e);
        }
        return EPrinterError.Unknown;
    }

    public EPrinterResponse askStatusJob(String name, String jobName) {
        try {
            Optional statusJob = this.commandHelper.askJobStatus(name, jobName);
            if (statusJob.isEmpty()) {
                this.logger.info("Job with ID {} for printer {} has status  EPrinterResponse.Unknown", (Object)jobName, (Object)name);
                return EPrinterResponse.Unknown;
            }
            EPrinterResponse ePrinterResponse = this.parseStatus((String)statusJob.get());
            this.logger.info("Job with ID {} for printer {} has status  EPrinterResponse.{}", new Object[]{jobName, name, ePrinterResponse.name()});
            return ePrinterResponse;
        }
        catch (IOException e) {
            this.logger.error("For {} printer EPrinterResponse is Unknown. askStatusJob command has error ", (Object)name, (Object)e);
            return EPrinterResponse.Unknown;
        }
    }

    private EPrinterResponse parseStatus(String status) {
        if (PRINTING.contains(status = status.trim().toLowerCase())) {
            return EPrinterResponse.Printing;
        }
        if (QUEUED.contains(status)) {
            return EPrinterResponse.In_Queue;
        }
        if (OFFLINE.contains(status)) {
            return EPrinterResponse.Offline;
        }
        if (CANCELED.contains(status)) {
            return EPrinterResponse.Canceled;
        }
        if (FAILED.contains(status)) {
            return EPrinterResponse.Failed;
        }
        if (COMPLETED.contains(status)) {
            return EPrinterResponse.Completed;
        }
        if (HAS_PROBLEM.contains(status)) {
            return EPrinterResponse.Has_Problem;
        }
        return EPrinterResponse.Unknown;
    }

    public long checkJobTime(String printerName, String jobName) {
        try {
            Optional jobStartingTime = this.commandHelper.getJobStartingTime(printerName, jobName);
            return jobStartingTime.map(zonedDateTime -> ChronoUnit.SECONDS.between((Temporal)zonedDateTime, ZonedDateTime.now())).orElse(-1L);
        }
        catch (Exception e) {
            this.logger.warn("Parse date error ", (Throwable)e);
            return -1L;
        }
    }

    public void cancelAllJobs() {
        try {
            this.commandHelper.cancelAllJobs();
        }
        catch (IOException e) {
            this.logger.warn("cancelAllJobs error ", (Throwable)e);
        }
    }

    public void restartCups() {
        try {
            this.commandHelper.restartCups();
        }
        catch (IOException e) {
            this.logger.error("Can not restart cups", (Throwable)e);
        }
    }

    public boolean clearAllJobs(String printerName) {
        try {
            return this.commandHelper.cancelAllJobs(printerName);
        }
        catch (IOException e) {
            this.logger.warn("cancelAllJobs error ", (Throwable)e);
            return false;
        }
    }

    public String getRunningJobs() {
        try {
            String str = this.commandHelper.getRunningJobsAsPlainText();
            if (StringUtils.hasText((String)str)) {
                return str;
            }
            return "---";
        }
        catch (IOException e) {
            return e.getLocalizedMessage();
        }
    }
}

